#ifndef __DM_PRODUCT_H__
#define __DM_PRODUCT_H__

/**
 *
 * \file        dm_product.h
 *
 * \brief       The purpose of this file is to allow code to be independent of
 *              a specific product, yet include things that are specific to a
 *              particular product.
 *
 * \author      Pete McCormick
 *
 * \date        04/21/2005
 *
 * \warning
 */

////////////////////////////////////////////////////////////////////////////////

#include "dm_nutypedefs.h"

////////////////////////////////////////////////////////////////////////////////
#ifdef __cplusplus
extern "C" {
#endif

#pragma pack(1)    // remove extra padding within structures.
#define SYSTEM_ID_LEN 12
typedef struct
{
    UINT32 pChecksum;    // pointer to checksum
    UINT32 pChecksumBegin; // beginning of checksum region
    UINT32 pChecksumEnd;   // end of checksum region
    UINT8 system_id[SYSTEM_ID_LEN];
    UINT16 major_rev;
    UINT16 minor_rev;
    UINT16 build;
    UINT8 cardId;
    UINT16 MaxFlash;    // max flash size in Kbytes
    UINT8 pad[3];       // fill to longword boundary
} FIRMWAREID;
#pragma pack()
extern const FIRMWAREID FirmwareId;
extern FIRMWAREID *pAppHeader;
////////////////////////////////////////////////////////////////////////////////

//New extended capability data structures
extern const char* pExtendedCapabilities[];
extern const UINT8 NumExtendedCapabilities;

extern char * ProductGetFirmwareId(void);
extern char * ProductGetLongName(UINT8 bStream);
char * ProductGetName(void);
extern char * ProductGetWelcomeMsg(void);
extern BOOL ProductHasUartConsole(void);
extern UINT32 ProductGetRemovableStorageDriveNum(void);
extern void ProductPrintInfo(void);
extern void ProductGetVersion(char * p, UINT32 len, char *cmd);
extern BOOL ProductFileIsTypeable(char * filename);
extern BOOL ProductHasRtc(void);
extern void FirmwareGetName(char * p, INT32 len, UINT8 bStream);
extern INT32 SelfTest(BOOL local);
extern void  ApplicationCleanup(void);
BOOL ProductHasSerialPort(UINT8 *pPacket, UINT32 slot);

// Capabilitiy reporting selection:
// when the the SYSTEM_ANALOG_CAPABILITIES_REPORT_SELECT join is received,
// respond with different information based on the value
enum
{
  SYSTEM_REPORT_SELECT_CAPABILITIES = 0,  // respond with capability serial
  SYSTEM_REPORT_SELECT_FIRMWARE_NAME,     // respond with firmware name serial
  SYSTEM_REPORT_SELECT_CARD_ID            // respond with card id analog
};

typedef enum
{
    CARD_ID_RES_ANALOG_A_AUDIO     = 1,   // Residential Analog card with analog audio
    CARD_ID_RES_ANALOG_SPDIF_AUDIO = 2,   // Residential Analog card with digital audio in
    CARD_ID_COM_ANALOG             = 3,   // Commerial Analog card with analog audio
    CARD_ID_DVI_RGBHV              = 4,   // DVI and RGBHV input card
    CARD_ID_ADV_HDMI               = 5,   // Advanced HDMI input card
    CARD_ID_BASIC_HDMI             = 6,   // Basic HDMI input card
    CARD_ID_WALLPLATE_300          = 7,   // DM Wallplate 300
    CARD_ID_ADV_DM                 = 8,   // Advanced Digital Media input card
    CARD_ID_BASIC_DM               = 9,   // Basic Digital Media input card
    CARD_ID_SECURITY_CAMERA        = 10,  // Security camera input card
    CARD_ID_HD_SDI                 = 11,  // SDI input card
    CARD_ID_ROOMBOX_4K_100_C       = 12,  // DM Roombox 4k 100-C (Valens)
    CARD_ID_ADV_FIBER_IN           = 13,  // Advanced Fiber input card
    CARD_ID_BASIC_FIBER_IN         = 14,  // Basic Fiber input card
    CARD_ID_WALLPLATE_300_F        = 15,  // DM Wallplate 300-F (fiber)
    CARD_ID_DMCO_CAT5_HD           = 16,  // DM CAT5 output card with local HDMI output
    CARD_ID_DMCO_CAT5              = 17,  // DM Output card without local HDMI
    CARD_ID_DMCO_HD                = 18,  // HDMI Output card
    CARD_ID_DMC_CO_HD              = 19,  // HDMI Output card (Valens)
    CARD_ID_DM_TX_400_3G           = 20,  // DM Wallplate 400-3G
    CARD_ID_WALLPLATE_200_2G       = 21,  // DM Wallplate 200-2G
    CARD_ID_DM_RX1_1G              = 22,  // DM-RX1-1G
    CARD_ID_DM_TX1_1G              = 23,  // DM-TX1-1G
    CARD_ID_DMCO_FIBER             = 24,  // DM fiber output card
    CARD_ID_ROOMBOX_100_C          = 25,  // DM Roombox 100-C (Valens)
    CARD_ID_ROOMBOX_100_F          = 26,  // DM Roombox 100-F (fiber)
    CARD_ID_WALLPLATE_100          = 27,  // DM Wallplate 100
    CARD_ID_WALLPLATE_200          = 28,  // DM Wallplate 200
    CARD_ID_WALLPLATE_300N         = 29,  // DM Wallplate 300N
    CARD_ID_WALLPLATE_100_F        = 30,  // DM Wallplate 100-F (fiber)
    CARD_ID_WALLPLATE_300N_F       = 31,  // DM Wallplate 300N-F (fiber)

// new card scheme
// note: when you add an output card add it to the IS_OUTPUT_CARD macro
// below as well!

    CARD_ID_DMPS_AUDIO_300         = 32,  // (00) DMPS AUDIO 300
    CARD_ID_WALLPLATE_201_C        = 33,  // (01) DM Wallplate 201-C
    CARD_ID_ROOMBOX_200            = 34,  // (02) DM Roombox 200
    CARD_ID_DMC_S                  = 35,  // (03) DM Input Card (Owlink)
    //CARD_ID_ROOMBOX_200_S          = 36,  // (04) DM Roombox 200 (Owlink)
    CARD_ID_ROOMBOX_100_S          = 37,  // (05) DM Roombox 100 (Owlink)
    CARD_ID_DMC_SO_HD              = 38,  // (06) DM Output Card (Owlink)
    CARD_ID_DMC_C_DSP              = 40,  // (08) DM Input Card (Valens)
    CARD_ID_DMC_C                  = 41,  // (09) DM Advanced Input Card (Valens)
    CARD_ID_DMC_S_DSP              = 42,  // (10) DM Advanced Input Card (Owlink)
    CARD_ID_VPG24                  = 43,  // (11) VPG24 (capacitive)
    CARD_ID_HD_SCALER              = 44,  // (12) HD Scaler
    CARD_ID_HD_DA2                 = 45,  // (13) HD-DA2
    CARD_ID_WALLPLATE_201_S        = 46,  // (14) DM Wallplate 201-S
    CARD_ID_RPS                    = 47,  // Power redundancy board
    CARD_ID_ROOMBOX_150_S          = 48,  // DM Roombox 150-s
    CARD_ID_DGE_1_GA               = 49,  // (17) DGE-1-GA (Valens)
    CARD_ID_DM_TX_401_C            = 50,  // (18) DM TX-401-C
    CARD_ID_WALLPLATE_401_S        = 52,  // (20) DM Wallplate 401-S
    CARD_ID_WALLPLATE_200_C_2G     = 53,  // (21) DM-TX-200-C-2G
    CARD_ID_DMC_S2O_HD             = 54,  // Single-mode owlink output card
    CARD_ID_DMC_S2_DSP             = 55,  // Single-mode owlink input card with DSP
    CARD_ID_DMC_S2                 = 56,  // Single-mode owlink input card without DSP
    CARD_ID_ROOMBOX_200_C_LT       = 57,  // (25) Lite version of RMC-200-C
    CARD_ID_FIBERTEST_TX           = 58,  // (26) DM Fiber Test Rx Card (Owlink)
    CARD_ID_FIBERTEST_RX           = 59,  // (27) DM Fiber Test Rx Card (Owlink)
    CARD_ID_DMC_VGA                = 61,  // (29) DMC-VGA
    CARD_ID_ROOMBOX_200_S          = 60,  // (28) RMC-200-S
    CARD_ID_SCALER_S               = 62,  // (30) RMC-SCALER-S

    CARD_ID_DM_TX1_C_1G            = 64,  // (32) DM_TX1-C-1G
    CARD_ID_DM_RX1_C_1G            = 65,  // (33) DM_RX1-C-1G
    CARD_ID_V24                    = 66,  // (34) V24 (resistive)
    CARD_ID_ROOMBOX_200_S2         = 67,  // (35) RMC-200-S2
    CARD_ID_SCALER_S2              = 68,  // (36) RMC-SCALER-S2
    CARD_ID_WALLPLATE_401_S2       = 70,   // (38) TX-401-S2
    CARD_ID_WALLPLATE_201_S2       = 71,  // (39) DM Wallplate 201-S2
    CARD_ID_WB_VID_TESTER          = 72,  // (40) DM Wide Band Video Tester
    CARD_ID_TSV_1051               = 73,  // (41) Lync 10" Touch Panel
    CARD_ID_CCS_UC_CODEC_AV        = 74,  // (42) Lync A/V board
    CARD_ID_HD_VIDEO_TESTER_S      = 75,  //  (43+32) Wideband video tester
    CARD_ID_HDXSPA                 = 77,  // HDXSPA
    CARD_ID_CCS_TX_201_C           = 78,  // (46) CCS-TX-201-C

    CARD_ID_DMC_4K_HD_DSP          = 80,  //(48) DMC-4K-HD-DSP (HDMI 4K input card with DSP)
    CARD_ID_DMC_4K_HD              = 81,  //(49) DMC-4K-HD (HDMI 4K input card)
    CARD_ID_DMC_TOC                = 82,  //(50) DMC-TIC - Test Input Card (HDMI)
    CARD_ID_DMC_TIC                = 83,  //(51) DMC-TIC - Test Input Card (HDMI)
    CARD_ID_DMB_4K_I_HD            = 84,  //(52) DMB-4K-I-HD - DM 4K Input blade

    CARD_ID_DMB_4K_O_C             = 86,  //(54) DMB-4K-O-C - DM 4K Ouput blade
    CARD_ID_DMCO_STRO              = 87,  // (55) DM Streaming Output card
    CARD_ID_DMC_4K_CO              = 88,  // (56) DM 4K Output card
    CARD_ID_TOB_300_2              = 89,  // (57) TOB-300-2 HDMI Dual Retimer Output Blade
    CARD_ID_DMC_STR                = 91,  // (59) DM Streaming Input card

    CARD_ID_REPEATER               = 100, // DM Repeater
    CARD_ID_ROOMBOX_100            = 101, // DM Roombox 100
                                          //
    CARD_ID_DM_MFS_10X1_IN         = 130, // (98) DM MFS input project
    CARD_ID_DM_MFS_10X1_OUT        = 131, // (99) DM MFS output project

    CARD_ID_C2N_IO                 = 198, // C2N-IO
    CARD_ID_HDXSP                  = 199, // HDXSP
    CARD_ID_MP_SERIES              = 200, // MP button series
    CARD_ID_IR_CONTROLLER          = 201, // IR Controller
    CARD_ID_PSAHD_7x400            = 202, // PSAHD-7x400
    CARD_ID_S3_IOP                 = 203, // Series 3 IO Processor
    CARD_ID_SWAMP_E                = 204, // SWAMP-E
    CARD_ID_DIN_DALI_2             = 205, // DIN DALI 2
    CARD_ID_PLX3                   = 206, // PLX3
    CARD_ID_FP                     = 207, // Front Panel board
    CARD_ID_DM_DGE2                = 208, // DM DGE-2
    CARD_ID_S3_IO_CARDS            = 209, // Series 3 IO Cards
    CARD_ID_GLS_EM_MCU             = 210, // GLS-EM-MCU energy monitor control board
    CARD_ID_S3_EXPAND              = 211, // Series 3 Expansion Chassis
    CARD_ID_CEN_SW_POE16           = 212, // POE switch and power suppply
    CARD_ID_TSX_IR_CONTROLLER      = 213, // IR Controller for TSx dock
    CARD_ID_DMC_FPTOUCH            = 214, // DM switch 64x64 touch controller
    CARD_ID_INET_IOEX              = 215, // INET-IOEX devices
    CARD_ID_CSA_PWS10S_HUB         = 216, // Cresnet hub
    CARD_ID_FLIP_TOP               = 217, // FlipTop series
    CARD_ID_TABLETOP               = 218, // TableTop Presentation Interface
    CARD_ID_GL_EXP                 = 219, // Dimmer with DALI interface
    CARD_ID_UNKNOWN                = 255  // Card Id is unknown for this device
} CARD_ID;

// update this when you add an output card!
#define IS_OUTPUT_CARD(cardId) (cardId == CARD_ID_DMCO_CAT5_HD || \
        cardId == CARD_ID_DMCO_CAT5 || \
        cardId == CARD_ID_DMCO_HD || \
        cardId == CARD_ID_DMC_CO_HD ||   \
        cardId == CARD_ID_DMCO_FIBER || \
        cardId == CARD_ID_DMC_SO_HD || \
        cardId == CARD_ID_DMC_S2O_HD || \
        cardId == CARD_ID_DMCO_STRO || \
        cardId == CARD_ID_DMC_4K_CO)

// Update this when you add a DM blade with ARM controllers
#define IS_DM_BLADE_W_ARM(cardId) (cardId == CARD_ID_DMB_4K_I_HD || \
        cardId == CARD_ID_DMB_4K_O_C)

// for bootloader Cnet support
#define HAS_FIXED_F0_CNET_ADDR(cardId) (cardId == CARD_ID_ROOMBOX_200 || \
        cardId == CARD_ID_ROOMBOX_200_S || \
        cardId == CARD_ID_SCALER_S || \
        cardId == CARD_ID_ROOMBOX_200_S2 || \
        cardId == CARD_ID_SCALER_S2 || \
        cardId == CARD_ID_ROOMBOX_150_S || \
        cardId == CARD_ID_ROOMBOX_100_S || \
        cardId == CARD_ID_ROOMBOX_100_C ||\
        cardId == CARD_ID_ROOMBOX_4K_100_C ||\
        cardId == CARD_ID_ROOMBOX_200   ||\
        cardId == CARD_ID_ROOMBOX_200_C_LT ||\
        cardId == CARD_ID_VPG24 ||\
        cardId == CARD_ID_V24 || \
        cardId == CARD_ID_TSV_1051 )

// has changeable cnet address stored in NVL
#define HAS_SETTABLE_CNET_ADDR(cardId) ((cardId == CARD_ID_WB_VID_TESTER) || (cardId == CARD_ID_C2N_IO))

// has external serial flash used to store the NVL
#define HAS_EXTERNAL_NVL_FLASH(cardId) ((cardId == CARD_ID_DMC_4K_HD) || \
                                        (cardId == CARD_ID_DMC_4K_HD_DSP) ||\
                                        (cardId == CARD_ID_DMC_4K_CO))

// the CAT5 Output card has a variant which does not have the local HDMI port
// we id that with its version number and than use a unique card id internally
#define DMCO_CAT5_NO_HDMI 0x40

// Srecord "address" for DM Firmware header
#define CARD_TYPE_ADDRESS 0xBAD0ADD0

// need to define constants used between the bootloader and the code for
// communications.  These MUST be consistent for both programs
// (functions defined in processor core)
UINT8 *GetAppFirmwareId(void);
UINT8 *GetApplicationBase(void);
UINT8 *GetBootCmdPtr(void);
UINT8 *GetBLBase(void);
UINT8 *GetPrebootBase(void);
UINT8 IsAddrInBootloader(UINT32);

#define UPGRADE_FIRMWARE      "UPG"
#define UPGRADE_FIRMWARE_ENET "UPE"
#define UPGRADE_FIRMWARE_USB  "UPU"
#define UPGRADE_NOTHING       "NOP"  // stay in bootloader
#define ENTER_MONITOR_USB     "MOU"  // stay in USB monitor
#define UPGRADE_CMD_LEN       4

//MP buttons use Cresnet
#if defined (SWAMP_E_SERIES)
#define DEFAULT_CRESNET_ADDRESS 0xB0  // default cresnet address for swamp-E
#define MIN_CRESNET_ADDRESS     0xB0
#define MAX_CRESNET_ADDRESS     0xB7
#elif defined(PSAHD_SERIES)
#define DEFAULT_CRESNET_ADDRESS 0xA2  // default cresnet address for PROAMP-7x400
#define MIN_CRESNET_ADDRESS     0xA2
#define MAX_CRESNET_ADDRESS     0xfe
#elif defined(CRESNET_ADDRESSING)
#define DEFAULT_CRESNET_ADDRESS 0x03  // default cresnet address for DM net
#define MIN_CRESNET_ADDRESS     0x03
#define MAX_CRESNET_ADDRESS     0xfe
#elif defined (DINDALI_SERIES)
#define DEFAULT_CRESNET_ADDRESS 0xA0  // default cresnet address for DIN_DALI_2 net
#define MIN_CRESNET_ADDRESS     0x03
#define MAX_CRESNET_ADDRESS     0xfe
#elif defined (C2NIO_SERIES)
#define DEFAULT_CRESNET_ADDRESS 0xA5  // default cresnet address for C2N-IO
#define MIN_CRESNET_ADDRESS     0x03
#define MAX_CRESNET_ADDRESS     0xfe
#elif defined (CN_HUB)
#define DEFAULT_CRESNET_ADDRESS 0x15  // default cresnet address for CSA-pWS10S-HUB
#define MIN_CRESNET_ADDRESS     0x03
#define MAX_CRESNET_ADDRESS     0xfe
#else // DM SERIES - DMnet
#define DEFAULT_CRESNET_ADDRESS 0xF0  // default cresnet address for DM net
#define DEFAULT_FRONT_PANEL_CRESNET_ADDRESS 0xFE
#define MIN_CRESNET_ADDRESS     0xb0
#define MAX_CRESNET_ADDRESS     0xCF
#endif

#define CRC_LENGTH 2

// Fatal error message for Bootloader.  It will send error join on Cresnet
// header (4 bytes), cnet address, subsystem, error
#define FATAL_ERROR_MSG "FEM"
#define FATAL_ERROR_LEN 4

// exception table length (at the end of RAM)
#define RAM_EXCEPTION_TABLE_LENGTH  8

// For binary image update (RF devices only at the moment)
#define FW_BIN_FILE_HDR_LEN_INITIAL    32
#define FW_BIN_FILE_HDR_VER_INITIAL    1

struct FIRMWARE_BIN_FILE_HEADER {
  UINT32  Crc;          // 32-bit CRC of main header contents only
  UINT16  Length;       // Length of the entire header
  UINT16  Ver;          // Header version
  UINT32  FileLen;      // Length of the entire file in bytes
  UINT8   FileVer[12];  // File version string
  UINT8   FileDate[4];  // Date the file was created (1 byte M, 1 byte D, 2 byte Y)
  UINT8   Reserved[4];  // Reserved for future use
};

#define FW_BIN_IMAGE_HDR_LEN_INITIAL   64
#define FW_BIN_IMAGE_HDR_VER_INITIAL    1

struct FIRMWARE_BIN_IMAGE_HEADER {
  UINT32  Crc;            // 32-bit CRC of main header contents only
  UINT16  Length;         // Header length in bytes
  UINT16  Ver;            // Header version
  UINT32  Flags;          // Option flags bit mask
  UINT16  ProductId;      // Product ID
  UINT8   TargetId;       // Target ID, device specific
  UINT8   Reserved1;      // Reserved/packing
  UINT8   VersionStr[14]; // Version string
  UINT16  MajorRev;       // Firmware build major revision, see DM build numbers
  UINT16  MinorRev;       // Firmware build minor revision, see DM build numbers
  UINT16  BuildNum;       // Firmware build numer, see DM build numbers
  UINT8   ImageDate[4];   // Date the file was created (1 byte M, 1 byte D, 2 byte Y)
  UINT32  ImageLen;       // Image length in bytes, does not include header
  UINT32  ImageCrc;       // 32-bit CRC of the image
  UINT8   Reserved2[16];  // Reserved for future use
};

enum FW_IMG_OPT_FLAGS {
  BYTE_ORDER_MSB    = 0x0001,   // If set then the byte order is MSB
  IMAGE_CRC_16_BIT  = 0x0002,  // If set, then the image uses 16 bit CRC
};

#ifdef __cplusplus
}
#endif

#endif

